/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_isend.c,v 1.98 2006/08/31 19:24:13 bgoglin Exp $";
 
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx_pin.h"
#include "mx__request.h"
#include "mx__segment.h"
#include "mx__lib.h"
#include "mx__self.h"
#include "mx__partner.h"
#include "mx__endpoint.h"
#include "mx__handle_map.h"
#include "mx__mcp_request_ring.h"
#include "mx__error.h"
#include "mx__shmem.h"

mx_return_t inline
mx__isend(struct mx_endpoint * ep,
	  mx_segment_t *segments_list, uint32_t segments_count,
	  mx_pin_type_t pin_type,
	  mx_endpoint_addr_t dest_address, uint64_t match_info,
	  void *context, mx_request_t *request)
{
  union mx_request *q = NULL;
  mcp_ureq_t *ureq;
  uint16_t mcp_handle;
  uint32_t length;
  mx_return_t ret_val;
  int small = 0;
  struct mx__partner * partner;

  if (unlikely(segments_count > MX_MAX_SEGMENTS)) {
    ret_val = MX_BAD_SEG_CNT;
    goto abort;
  }

  ret_val = MX_SUCCESS;

  q = mx__rl_alloc(ep);
  if (unlikely(q == NULL)) {
    ret_val = MX_NO_RESOURCES;
    goto abort;
  }

  MX__CACHE_SEGMENTS(q->send, segments_list, segments_count);
  if (unlikely((q->send.count > 1) && (q->send.segments == NULL))) {
    ret_val = MX_NO_RESOURCES;
    goto abort;
  }
#ifndef MX_KERNEL
  if (unlikely(MX_DEBUG_CSUM && mx__opt.csum)) {
    mx_always_assert(match_info >> 48 == 0);
    if (q->send.count)
      match_info |= 
	(uint64_t)mx_checksum(q->send.segments[0].segment_ptr, 
			      q->send.segments[0].segment_length) << 48;
  }
#endif
#if defined __GNUC__ && MX_DEBUG
  q->send.caller = __builtin_return_address(0);
#endif
  q->send.memory_context = pin_type;

  partner = mx__partner_from_addr(&dest_address);
  q->basic.partner = partner;
  q->send.basic.status.source = dest_address;
  q->send.basic.status.match_info = match_info;
#if MX_DEBUG
  q->send.basic.status.xfer_length = -1;
#endif
  q->send.basic.status.context = context;
  q->send.basic.mcp_handle = -1;
  q->send.basic.state = MX__REQUEST_STATE_SEND_QUEUED;
  q->send.basic.wq = NULL;
  q->send.basic.acquired_by_wait_any = 0;
  length = mx__total_segment_length(q->send.segments, q->send.count);
  q->send.basic.status.msg_length = length;
  if (likely(length <= ep->tiny_msg_threshold)) {
    q->send.basic.type = MX__REQUEST_TYPE_SEND_TINY;
    MX__EP_STATS_INC(ep, isend_tiny);
    small = 1;
  } else if (length <= ep->small_msg_threshold) {
    q->send.basic.type = MX__REQUEST_TYPE_SEND_SMALL;
    MX__EP_STATS_INC(ep, isend_small);
    small = 1;
  } else if (length <= ep->medium_msg_threshold) {
    q->send.basic.type = MX__REQUEST_TYPE_SEND_MEDIUM;
    MX__EP_STATS_INC(ep, isend_medium);
  } else {
    q->send.basic.type = MX__REQUEST_TYPE_SEND_LARGE;
    MX__EP_STATS_INC(ep, isend_large);
  }
  q->basic.requeued = 0;

#ifndef MX_KERNEL
  if (unlikely(partner == ep->myself && !mx__opt.disable_self)) {
    mx__self_send(ep, q);
  } else
#if MX_USE_SHMEM
    if (unlikely(partner->peer_index_n == ep->myself->peer_index_n
		 && !mx__opt.disable_shmem)) {
    mx__shm_send(ep,q);
  } else
#endif
#endif
    if (likely(small
	       && mx__isempty_request_queue(&ep->send_reqq)
	       && mx__endpoint_avail_mcp_handle(ep))) {
    q->send.basic.state = MX__REQUEST_STATE_MCP;
    mcp_handle = mx__endpoint_alloc_mcp_handle(ep);
    ureq = MX__UREQ(ep, mcp_handle);
    mx__hm_set(ep->handle_map, mcp_handle, q);
    mx__post_send(ep, q, ureq, mcp_handle);
  } else {
    mx__enqueue_request(&ep->send_reqq, q);
  }

  mx__luigi(ep);

  *request = q;
  return MX_SUCCESS;

 abort:
  if (q) {
    mx__rl_free(ep, q);
  }
  return mx__error(ep, "mx_isend", ret_val);
}

#ifndef MX_KERNEL
MX_FUNC(mx_return_t)
mx_isend(struct mx_endpoint * ep,
	 mx_segment_t *segments_list, uint32_t segments_count,
	 mx_endpoint_addr_t dest_address, uint64_t match_info,
	 void *context, mx_request_t *request)
{
  mx_return_t rc;
#if MX_ONE_SIDED
  mx_always_assert(!(match_info & MX__ONESIDED_MASK));
#endif
  MX__MUTEX_LOCK(&ep->lock);
  rc = mx__isend(ep, segments_list, segments_count, MX_PIN_UNDEFINED,
		 dest_address, match_info, context, request);
  MX__MUTEX_UNLOCK(&ep->lock);
  return rc;
}
#else
MX_FUNC(mx_return_t)
mx_kisend(struct mx_endpoint * ep,
	 mx_segment_t *segments_list, uint32_t segments_count,
	 mx_pin_type_t pin_type,
	 mx_endpoint_addr_t dest_address, uint64_t match_info,
	 void *context, mx_request_t *request)
{
  mx_return_t rc;
  MX__MUTEX_LOCK(&ep->lock);
  rc = mx__isend(ep, segments_list, segments_count, pin_type,
		 dest_address, match_info, context, request);
  MX__MUTEX_UNLOCK(&ep->lock);
  return rc;
}
#endif
